import chalk from 'chalk';
import { stripe } from './stripe/client.js';
import { openai } from './chatgpt/client.js';
import { env } from './config/env.js';

async function fetchTodayPayments() {
  const today = new Date();
  today.setHours(0, 0, 0, 0);
  const timestamp = Math.floor(today.getTime() / 1000);

  const payments = await stripe.paymentIntents.list({
    created: { gte: timestamp },
    limit: 100,
  });

  return payments.data.filter((p) => p.status === 'succeeded');
}

async function main() {
  console.log(chalk.bold.cyan('\n📊 Stripe 매출 데이터 분석\n'));
  console.log(chalk.gray('='.repeat(60) + '\n'));

  // 1. 데이터 조회
  console.log(chalk.blue('📥 Stripe 데이터 조회 중...\n'));
  const payments = await fetchTodayPayments();

  if (payments.length === 0) {
    console.log(chalk.yellow('⚠️  오늘 거래가 없습니다.'));
    console.log(chalk.gray('먼저 결제를 생성하세요: npm run generate\n'));
    return;
  }

  // 2. 통계 계산
  const totalAmount = payments.reduce((sum, p) => sum + p.amount, 0);
  const totalCount = payments.length;
  const averageAmount = totalAmount / totalCount;

  console.log(chalk.bold('=== 기본 통계 ===\n'));
  console.log(`총 매출: ${chalk.green(`$${(totalAmount / 100).toFixed(2)}`)}`);
  console.log(`결제 건수: ${chalk.white(totalCount + '건')}`);
  console.log(`평균 금액: ${chalk.cyan(`$${(averageAmount / 100).toFixed(2)}`)}`);

  // 3. ChatGPT 분석
  console.log(chalk.gray('\n' + '='.repeat(60)));
  console.log(chalk.bold.cyan('\n🤖 ChatGPT 분석 중...\n'));

  const dataText = `
날짜: ${new Date().toISOString().split('T')[0]}
총 매출: $${(totalAmount / 100).toFixed(2)}
결제 건수: ${totalCount}건
평균 결제 금액: $${(averageAmount / 100).toFixed(2)}

개별 결제:
${payments.slice(0, 10).map((p, i) =>
  `${i + 1}. $${(p.amount / 100).toFixed(2)} - ${p.description || 'No description'}`
).join('\n')}
${payments.length > 10 ? `\n... 외 ${payments.length - 10}건` : ''}
  `;

  const prompt = `다음은 오늘의 매출 데이터입니다:

${dataText}

이 데이터를 분석하여 다음을 제공해주세요:

## 📊 매출 요약
(2-3문장으로 요약)

## 💡 주요 인사이트
(3가지)

## 📈 개선 제안
(2가지)

간결하고 실용적으로 작성해주세요.`;

  const response = await openai.chat.completions.create({
    model: env.openaiModel,
    messages: [
      {
        role: 'system',
        content: '당신은 전문적인 비즈니스 애널리스트입니다. 매출 데이터를 분석하여 실용적인 인사이트를 제공합니다.',
      },
      {
        role: 'user',
        content: prompt,
      },
    ],
    max_tokens: env.openaiMaxTokens,
    temperature: env.openaiTemperature,
  });

  const analysis = response.choices[0].message.content || '';

  console.log(chalk.gray('='.repeat(60)));
  console.log(chalk.bold.cyan('\n🎯 분석 결과\n'));
  console.log(chalk.gray('='.repeat(60) + '\n'));
  console.log(chalk.white(analysis));
  console.log(chalk.gray('\n' + '='.repeat(60)));
  console.log(chalk.bold.green('\n✨ 분석 완료!\n'));
}

main().catch((error) => {
  console.error(chalk.red('\n💥 에러 발생:'));
  console.error(error);
  process.exit(1);
});
